<?php

namespace Bixo\Payroll\Http\Controllers;

use Exception;
use Bixo\Payroll\Actions\SalaryWorkflow;
use Bixo\Payroll\Http\Requests\SalaryWorkflowRequest;
use Bixo\Payroll\Http\Resources\SalaryResource;
use Bixo\Payroll\Models\Salary;
use Litepie\Http\Controllers\ActionController as BaseController;

/**
 * Workflow controller class.
 *
 */
class SalaryWorkflowController extends BaseController
{
    /**
     * Action controller function for salary.
     *
     * @param Model $salary
     * @param action next action for the salary.
     *
     * @return Response
     */
    public function __invoke(SalaryWorkflowRequest $request, Salary $salary, $trasition)
    {
        try {
            $request = $request->all();
            $salary = SalaryWorkflow::run($trasition, $salary, $request);
            $data = new SalaryResource($salary);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('payroll::salary.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('payroll/salary/' . $salary->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('payroll/salary/' . $salary->getRouteKey()))
                ->redirect();
        }
    }


    /**
     * Get the salary and user for the workflow.
     *
     * @param PublicRequest $request
     * @param string $salary
     * @param string $user
     *
     * @return Response
     */
    public function get(PublicRequest $request, $salary, $user)
    {
        if (!$request->hasValidSignature()) {
            abort(403, 'This url is expired!');
        }

        $auth = Auth::guard()->getProvider()->getModel();
        $user = $auth::findBySignedId($user);
        Auth::guard()->login($user);
        $salary = Salary::findBySignedId($salary);
        $transitions = $this->transitions($salary, $user);
        $user = user();
        Auth::logout();
        return $this->response->setMetaTitle(trans('payroll::salary.name') . ' ' . trans('Approvals'))
            ->view('payroll::public.salary.workflow')
            ->layout('mobile')
            ->data(compact('salary', 'user', 'transitions'))
            ->output();
    }

    /**
     * Display the workflow form for the salary.
     *
     * @param PublicRequest $request
     * @param string $salary
     * @param string $user
     *
     * @return Response
     */
    public function post(PublicRequest $request, $salary, $user)
    {
        if (!$request->hasValidSignature()) {
            abort(403, 'This url is expired!');
        }
        $transition = $request->transition;
        $auth = Auth::guard()->getProvider()->getModel();
        $user = $auth::findBySignedId($user);
        Auth::guard()->login($user);
        $salary = Salary::findBySignedId($salary);
        $user = user();
        $salary = SalaryWorkflow::run($transition, $salary, $request->all());
        Auth::logout();

        return response()->json(
            [
                'status' => 'success',
                'url' => $request->url(),
            ]
        );
    }

    private function transitions($salary, $user)
    {
        $transitions = [];
        foreach ($salary->workflow()->transitions($salary) as $key => $value) {
            $name = $value->getName();
            $array['url'] = URL::temporarySignedRoute('litecms.salary.workflow', now()->addMinutes(3), [
                'guard' => 'admin',
                'transition' => $name,
                'salary' => $salary->getSignedId(now()->addMinutes(3)),
                'user' => $user->getSignedId(now()->addMinutes(3)),
                'trans' => 'en',
            ]
            );
            $array['name'] = $name;
            $array['key'] = $name;
            $array['form'] = $salary->workflow()->form($value);
            $array['label'] = trans('payroll::salary.workflow.' . $name);
            $transitions[] = $array;
        }
        return $transitions;
    }
}
