<?php

namespace Bixo\Payroll\Http\Controllers;

use Exception;
use Litepie\Http\Controllers\ResourceController as BaseController;
use Bixo\Payroll\Forms\Slab as SlabForm;
use Bixo\Payroll\Http\Requests\SlabResourceRequest;
use Bixo\Payroll\Http\Resources\SlabResource;
use Bixo\Payroll\Http\Resources\SlabsCollection;
use Bixo\Payroll\Models\Slab;
use Bixo\Payroll\Actions\SlabAction;
use Bixo\Payroll\Actions\SlabActions;


/**
 * Resource controller class for slab.
 */
class SlabResourceController extends BaseController
{

    /**
     * Initialize slab resource controller.
     *
     *
     * @return null
     */
    public function __construct()
    {
        parent::__construct();
        $this->middleware(function ($request, $next) {
            $this->form = SlabForm::grouped(false)
                            ->setAttributes()
                            ->toArray();
            $this->modules = $this->modules(config('bixo.payroll.modules'), 'payroll', guard_url('payroll'));
            return $next($request);
        });
    }

    /**
     * Display a list of slab.
     *
     * @return Response
     */
    public function index(SlabResourceRequest $request)
    {
        $request = $request->all();
        $page = SlabActions::run('paginate', $request);

        $data = new SlabsCollection($page);

        $form = $this->form;
        $modules = $this->modules;

        return $this->response->setMetaTitle(trans('payroll::slab.names'))
            ->view('payroll::slab.index')
            ->data(compact('data', 'modules', 'form'))
            ->output();

    }

    /**
     * Display slab.
     *
     * @param Request $request
     * @param Model   $slab
     *
     * @return Response
     */
    public function show(SlabResourceRequest $request, Slab $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new SlabResource($model);
        return $this->response
            ->setMetaTitle(trans('app.view') . ' ' . trans('payroll::slab.name'))
            ->data(compact('data', 'form', 'modules'))
            ->view('payroll::slab.show')
            ->output();
    }

    /**
     * Show the form for creating a new slab.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(SlabResourceRequest $request, Slab $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new SlabResource($model);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('payroll::slab.name'))
            ->view('payroll::slab.create')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Create new slab.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(SlabResourceRequest $request, Slab $model)
    {
        try {
            $request = $request->all();
            $model = SlabAction::run('store', $model, $request);
            $data = new SlabResource($model);
            return $this->response->message(trans('messages.success.created', ['Module' => trans('payroll::slab.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('payroll/slab/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/payroll/slab'))
                ->redirect();
        }

    }

    /**
     * Show slab for editing.
     *
     * @param Request $request
     * @param Model   $slab
     *
     * @return Response
     */
    public function edit(SlabResourceRequest $request, Slab $model)
    {
        $form = $this->form;
        $modules = $this->modules;
        $data = new SlabResource($model);
        // return view('payroll::slab.edit', compact('data', 'form', 'modules'));

        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('payroll::slab.name'))
            ->view('payroll::slab.edit')
            ->data(compact('data', 'form', 'modules'))
            ->output();

    }

    /**
     * Update the slab.
     *
     * @param Request $request
     * @param Model   $slab
     *
     * @return Response
     */
    public function update(SlabResourceRequest $request, Slab $model)
    {
        try {
            $request = $request->all();
            $model = SlabAction::run('update', $model, $request);
            $data = new SlabResource($model);

            return $this->response->message(trans('messages.success.updated', ['Module' => trans('payroll::slab.name')]))
                ->code(204)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('payroll/slab/' . $model->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('payroll/slab/' .  $model->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the slab.
     *
     * @param Model   $slab
     *
     * @return Response
     */
    public function destroy(SlabResourceRequest $request, Slab $model)
    {
        try {

            $request = $request->all();
            $model = SlabAction::run('destroy', $model, $request);
            $data = new SlabResource($model);

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('payroll::slab.name')]))
                ->code(202)
                ->status('success')
                ->data(compact('data'))
                ->url(guard_url('payroll/slab/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('payroll/slab/' .  $model->getRouteKey()))
                ->redirect();
        }

    }
}
