<?php

namespace Bixo\Payroll\Http\Controllers;

use Exception;
use Bixo\Payroll\Actions\SlabWorkflow;
use Bixo\Payroll\Http\Requests\SlabWorkflowRequest;
use Bixo\Payroll\Http\Resources\SlabResource;
use Bixo\Payroll\Models\Slab;
use Litepie\Http\Controllers\ActionController as BaseController;

/**
 * Workflow controller class.
 *
 */
class SlabWorkflowController extends BaseController
{
    /**
     * Action controller function for slab.
     *
     * @param Model $slab
     * @param action next action for the slab.
     *
     * @return Response
     */
    public function __invoke(SlabWorkflowRequest $request, Slab $slab, $trasition)
    {
        try {
            $request = $request->all();
            $slab = SlabWorkflow::run($trasition, $slab, $request);
            $data = new SlabResource($slab);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('payroll::slab.name')]))
                ->code(204)
                ->data(compact('data'))
                ->status('success')
                ->url(guard_url('payroll/slab/' . $slab->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('payroll/slab/' . $slab->getRouteKey()))
                ->redirect();
        }
    }


    /**
     * Get the slab and user for the workflow.
     *
     * @param PublicRequest $request
     * @param string $slab
     * @param string $user
     *
     * @return Response
     */
    public function get(PublicRequest $request, $slab, $user)
    {
        if (!$request->hasValidSignature()) {
            abort(403, 'This url is expired!');
        }

        $auth = Auth::guard()->getProvider()->getModel();
        $user = $auth::findBySignedId($user);
        Auth::guard()->login($user);
        $slab = Slab::findBySignedId($slab);
        $transitions = $this->transitions($slab, $user);
        $user = user();
        Auth::logout();
        return $this->response->setMetaTitle(trans('payroll::slab.name') . ' ' . trans('Approvals'))
            ->view('payroll::public.slab.workflow')
            ->layout('mobile')
            ->data(compact('slab', 'user', 'transitions'))
            ->output();
    }

    /**
     * Display the workflow form for the slab.
     *
     * @param PublicRequest $request
     * @param string $slab
     * @param string $user
     *
     * @return Response
     */
    public function post(PublicRequest $request, $slab, $user)
    {
        if (!$request->hasValidSignature()) {
            abort(403, 'This url is expired!');
        }
        $transition = $request->transition;
        $auth = Auth::guard()->getProvider()->getModel();
        $user = $auth::findBySignedId($user);
        Auth::guard()->login($user);
        $slab = Slab::findBySignedId($slab);
        $user = user();
        $slab = SlabWorkflow::run($transition, $slab, $request->all());
        Auth::logout();

        return response()->json(
            [
                'status' => 'success',
                'url' => $request->url(),
            ]
        );
    }

    private function transitions($slab, $user)
    {
        $transitions = [];
        foreach ($slab->workflow()->transitions($slab) as $key => $value) {
            $name = $value->getName();
            $array['url'] = URL::temporarySignedRoute('litecms.slab.workflow', now()->addMinutes(3), [
                'guard' => 'admin',
                'transition' => $name,
                'slab' => $slab->getSignedId(now()->addMinutes(3)),
                'user' => $user->getSignedId(now()->addMinutes(3)),
                'trans' => 'en',
            ]
            );
            $array['name'] = $name;
            $array['key'] = $name;
            $array['form'] = $slab->workflow()->form($value);
            $array['label'] = trans('payroll::slab.workflow.' . $name);
            $transitions[] = $array;
        }
        return $transitions;
    }
}
