<?php

namespace Bixo\Payroll\Listeners;

use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;
use Bixo\Payroll\Events\AdjustmentWorkflow as AdjustmentWorkflowEvent;
use Bixo\Payroll\Notifications\AdjustmentWorkflow as AdjustmentWorkflowNotification;
use Litepie\Actions\Concerns\AsAction;

class AdjustmentWorkflow
{
    use AsAction;

    private $allowedTransitions = [
        'before' => [],
        'after' => ['publish', 'submit', 'approve'],
    ];

    public function handle(AdjustmentWorkflowEvent $event)
    {
        $function = Str::camel($event->transition);
        return $this->$function($event);
    }

    /**
     * Sends a notification to the client when the $adjustment is submitted.
     *
     * @param AdjustmentWorkflowEvent $event The event object.
     * @return void
     */
    public function submit(AdjustmentWorkflowEvent $event)
    {
        $client = $event->adjustment->client;
        Notification::send($client, new AdjustmentWorkflowNotification($event));
    }

    /**
     * Sends a notification to the client when the $adjustment is published.
     *
     * @param AdjustmentWorkflowEvent $event The event object.
     * @return void
     */
    public function publish(AdjustmentWorkflowEvent $event)
    {
        $client = $event->adjustment->client;
        Notification::send($client, new AdjustmentWorkflowNotification($event));
    }

    /**
     * Sends a notification to the client when the $adjustment is approved.
     *
     * @param AdjustmentWorkflowEvent $event The event object.
     * @return void
     */
    public function approve(AdjustmentWorkflowEvent $event)
    {
        $client = $event->adjustment->client;
        Notification::send($client, new AdjustmentWorkflowNotification($event));
    }

    /**
     * Handles the $adjustment workflow event as a listener.
     *
     * @param AdjustmentWorkflowEvent $event The event object.
     * @return void
     */
    public function asListener(AdjustmentWorkflowEvent $event)
    {
        if (($event->when == 'before' &&
            in_array($event->transition, $this->allowedTransitions['before']) ||
            $event->when == 'after' &&
            in_array($event->transition, $this->allowedTransitions['after']))
        ) {
            return $this->handle($event);
        }
    }
}
