<?php

namespace Bixo\Payroll\Listeners;

use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;
use Bixo\Payroll\Events\CommissionWorkflow as CommissionWorkflowEvent;
use Bixo\Payroll\Notifications\CommissionWorkflow as CommissionWorkflowNotification;
use Litepie\Actions\Concerns\AsAction;

class CommissionWorkflow
{
    use AsAction;

    private $allowedTransitions = [
        'before' => [],
        'after' => ['publish', 'submit', 'approve'],
    ];

    public function handle(CommissionWorkflowEvent $event)
    {
        $function = Str::camel($event->transition);
        return $this->$function($event);
    }

    /**
     * Sends a notification to the client when the $commission is submitted.
     *
     * @param CommissionWorkflowEvent $event The event object.
     * @return void
     */
    public function submit(CommissionWorkflowEvent $event)
    {
        $client = $event->commission->client;
        Notification::send($client, new CommissionWorkflowNotification($event));
    }

    /**
     * Sends a notification to the client when the $commission is published.
     *
     * @param CommissionWorkflowEvent $event The event object.
     * @return void
     */
    public function publish(CommissionWorkflowEvent $event)
    {
        $client = $event->commission->client;
        Notification::send($client, new CommissionWorkflowNotification($event));
    }

    /**
     * Sends a notification to the client when the $commission is approved.
     *
     * @param CommissionWorkflowEvent $event The event object.
     * @return void
     */
    public function approve(CommissionWorkflowEvent $event)
    {
        $client = $event->commission->client;
        Notification::send($client, new CommissionWorkflowNotification($event));
    }

    /**
     * Handles the $commission workflow event as a listener.
     *
     * @param CommissionWorkflowEvent $event The event object.
     * @return void
     */
    public function asListener(CommissionWorkflowEvent $event)
    {
        if (($event->when == 'before' &&
            in_array($event->transition, $this->allowedTransitions['before']) ||
            $event->when == 'after' &&
            in_array($event->transition, $this->allowedTransitions['after']))
        ) {
            return $this->handle($event);
        }
    }
}
