<?php

namespace Bixo\Payroll\Listeners;

use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;
use Bixo\Payroll\Events\PayrollAction as PayrollActionEvent;
use Bixo\Payroll\Notifications\PayrollAction as PayrollActionNotification;
use Litepie\Actions\Concerns\AsAction;

class PayrollAction
{
    use AsAction;

    private $allowedActions = [
        'before' => [],
        'after' => ['create'],
    ];

    /**
     * Handle the PayrollActionEvent.
     *
     * @param   PayrollActionEvent  $event
     * @return mixed
     */
    public function handle(PayrollActionEvent $event)
    {
        $function = Str::camel($event->action);
        return $this->$function($event);
    }

    /**
     * Create a new $payroll.
     *
     * @param   PayrollActionEvent  $event
     * @return void
     */
    public function create(PayrollActionEvent $event)
    {
        $client = $event->payroll->client;
        Notification::send($client, new PayrollActionNotification($event));
    }

    /**
     * Handle the PayrollActionEvent as a listener.
     *
     * @param   PayrollActionEvent  $event
     * @return mixed
     */
    public function asListener(PayrollActionEvent $event)
    {
        if ($this->isAllowed($event)) {
            return $this->handle($event);
        }
    }

    /**
     * Check if the event action is allowed.
     *
     * @param   PayrollActionEvent  $event
     * @return bool
     */
    private function isAllowed(PayrollActionEvent $event)
    {
        if ($event->when == 'before' &&
            !in_array($event->action, $this->allowedActions['before'])) {
            return false;
        }

        if (($event->when == 'after' &&
            !in_array($event->action, $this->allowedActions['after']))
        ) {
            return false;
        }

        return true;
    }
}
