<?php

namespace Bixo\Payroll\Listeners;

use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;
use Bixo\Payroll\Events\PayrollWorkflow as PayrollWorkflowEvent;
use Bixo\Payroll\Notifications\PayrollWorkflow as PayrollWorkflowNotification;
use Litepie\Actions\Concerns\AsAction;

class PayrollWorkflow
{
    use AsAction;

    private $allowedTransitions = [
        'before' => [],
        'after' => ['publish', 'submit', 'approve'],
    ];

    public function handle(PayrollWorkflowEvent $event)
    {
        $function = Str::camel($event->transition);
        return $this->$function($event);
    }

    /**
     * Sends a notification to the client when the $payroll is submitted.
     *
     * @param PayrollWorkflowEvent $event The event object.
     * @return void
     */
    public function submit(PayrollWorkflowEvent $event)
    {
        $client = $event->payroll->client;
        Notification::send($client, new PayrollWorkflowNotification($event));
    }

    /**
     * Sends a notification to the client when the $payroll is published.
     *
     * @param PayrollWorkflowEvent $event The event object.
     * @return void
     */
    public function publish(PayrollWorkflowEvent $event)
    {
        $client = $event->payroll->client;
        Notification::send($client, new PayrollWorkflowNotification($event));
    }

    /**
     * Sends a notification to the client when the $payroll is approved.
     *
     * @param PayrollWorkflowEvent $event The event object.
     * @return void
     */
    public function approve(PayrollWorkflowEvent $event)
    {
        $client = $event->payroll->client;
        Notification::send($client, new PayrollWorkflowNotification($event));
    }

    /**
     * Handles the $payroll workflow event as a listener.
     *
     * @param PayrollWorkflowEvent $event The event object.
     * @return void
     */
    public function asListener(PayrollWorkflowEvent $event)
    {
        if (($event->when == 'before' &&
            in_array($event->transition, $this->allowedTransitions['before']) ||
            $event->when == 'after' &&
            in_array($event->transition, $this->allowedTransitions['after']))
        ) {
            return $this->handle($event);
        }
    }
}
