<?php

namespace Bixo\Payroll\Listeners;

use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;
use Bixo\Payroll\Events\SalaryWorkflow as SalaryWorkflowEvent;
use Bixo\Payroll\Notifications\SalaryWorkflow as SalaryWorkflowNotification;
use Litepie\Actions\Concerns\AsAction;

class SalaryWorkflow
{
    use AsAction;

    private $allowedTransitions = [
        'before' => [],
        'after' => ['publish', 'submit', 'approve'],
    ];

    public function handle(SalaryWorkflowEvent $event)
    {
        $function = Str::camel($event->transition);
        return $this->$function($event);
    }

    /**
     * Sends a notification to the client when the $salary is submitted.
     *
     * @param SalaryWorkflowEvent $event The event object.
     * @return void
     */
    public function submit(SalaryWorkflowEvent $event)
    {
        $client = $event->salary->client;
        Notification::send($client, new SalaryWorkflowNotification($event));
    }

    /**
     * Sends a notification to the client when the $salary is published.
     *
     * @param SalaryWorkflowEvent $event The event object.
     * @return void
     */
    public function publish(SalaryWorkflowEvent $event)
    {
        $client = $event->salary->client;
        Notification::send($client, new SalaryWorkflowNotification($event));
    }

    /**
     * Sends a notification to the client when the $salary is approved.
     *
     * @param SalaryWorkflowEvent $event The event object.
     * @return void
     */
    public function approve(SalaryWorkflowEvent $event)
    {
        $client = $event->salary->client;
        Notification::send($client, new SalaryWorkflowNotification($event));
    }

    /**
     * Handles the $salary workflow event as a listener.
     *
     * @param SalaryWorkflowEvent $event The event object.
     * @return void
     */
    public function asListener(SalaryWorkflowEvent $event)
    {
        if (($event->when == 'before' &&
            in_array($event->transition, $this->allowedTransitions['before']) ||
            $event->when == 'after' &&
            in_array($event->transition, $this->allowedTransitions['after']))
        ) {
            return $this->handle($event);
        }
    }
}
