<?php

namespace Bixo\Payroll\Listeners;

use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Str;
use Bixo\Payroll\Events\SlabWorkflow as SlabWorkflowEvent;
use Bixo\Payroll\Notifications\SlabWorkflow as SlabWorkflowNotification;
use Litepie\Actions\Concerns\AsAction;

class SlabWorkflow
{
    use AsAction;

    private $allowedTransitions = [
        'before' => [],
        'after' => ['publish', 'submit', 'approve'],
    ];

    public function handle(SlabWorkflowEvent $event)
    {
        $function = Str::camel($event->transition);
        return $this->$function($event);
    }

    /**
     * Sends a notification to the client when the $slab is submitted.
     *
     * @param SlabWorkflowEvent $event The event object.
     * @return void
     */
    public function submit(SlabWorkflowEvent $event)
    {
        $client = $event->slab->client;
        Notification::send($client, new SlabWorkflowNotification($event));
    }

    /**
     * Sends a notification to the client when the $slab is published.
     *
     * @param SlabWorkflowEvent $event The event object.
     * @return void
     */
    public function publish(SlabWorkflowEvent $event)
    {
        $client = $event->slab->client;
        Notification::send($client, new SlabWorkflowNotification($event));
    }

    /**
     * Sends a notification to the client when the $slab is approved.
     *
     * @param SlabWorkflowEvent $event The event object.
     * @return void
     */
    public function approve(SlabWorkflowEvent $event)
    {
        $client = $event->slab->client;
        Notification::send($client, new SlabWorkflowNotification($event));
    }

    /**
     * Handles the $slab workflow event as a listener.
     *
     * @param SlabWorkflowEvent $event The event object.
     * @return void
     */
    public function asListener(SlabWorkflowEvent $event)
    {
        if (($event->when == 'before' &&
            in_array($event->transition, $this->allowedTransitions['before']) ||
            $event->when == 'after' &&
            in_array($event->transition, $this->allowedTransitions['after']))
        ) {
            return $this->handle($event);
        }
    }
}
