<?php

namespace Litecms\Quote\Http\Controllers;

use App\Http\Controllers\APIController as BaseController;
use Litecms\Quote\Http\Requests\QuotesRequest;
use Litecms\Quote\Interfaces\QuotesRepositoryInterface;
use Litecms\Quote\Models\Quotes;
use Litecms\Quote\Forms\Quotes as Form;

/**
 * APIController  class for quotes.
 */
class QuotesAPIController extends BaseController
{

    /**
     * Initialize quotes resource controller.
     *
     * @param type QuotesRepositoryInterface $quotes
     *
     * @return null
     */
    public function __construct(QuotesRepositoryInterface $quotes)
    {
        parent::__construct();
        $this->repository = $quotes;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Litecms\Quote\Repositories\Criteria\QuotesResourceCriteria::class);
    }

    /**
     * Display a list of quotes.
     *
     * @return Response
     */
    public function index(QuotesRequest $request)
    {
        return $this->repository
            ->setPresenter(\Litecms\Quote\Repositories\Presenter\QuotesPresenter::class)
            ->paginate();
    }

    /**
     * Display quotes.
     *
     * @param Request $request
     * @param Model   $quotes
     *
     * @return Response
     */
    public function show(QuotesRequest $request, Quotes $quotes)
    {
        return $quotes->setPresenter(\Litecms\Quote\Repositories\Presenter\QuotesListPresenter::class);
        ;
    }

    /**
     * Create new quotes.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(QuotesRequest $request)
    {
        try {
            $data              = $request->all();
            $data['user_id']   = user_id();
            $data['user_type'] = user_type();
            $data              = $this->repository->create($data);
            $message           = trans('messages.success.created', ['Module' => trans('quote::quotes.name')]);
            $code              = 204;
            $status            = 'success';
            $url               = guard_url('quote/quotes/' . $quotes->getRouteKey());
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('quote/quotes');
        }
        return compact('data', 'message', 'code', 'status', 'url');
    }

    /**
     * Update the quotes.
     *
     * @param Request $request
     * @param Model   $quotes
     *
     * @return Response
     */
    public function update(QuotesRequest $request, Quotes $quotes)
    {
        try {
            $data = $request->all();

            $quotes->update($data);
            $message = trans('messages.success.updated', ['Module' => trans('quote::quotes.name')]);
            $code    = 204;
            $status  = 'success';
            $url     = guard_url('quote/quotes/' . $quotes->getRouteKey());
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('quote/quotes/' . $quotes->getRouteKey());
        }
        return compact('data', 'message', 'code', 'status', 'url');
    }

    /**
     * Remove the quotes.
     *
     * @param Model   $quotes
     *
     * @return Response
     */
    public function destroy(QuotesRequest $request, Quotes $quotes)
    {
        try {
            $quotes->delete();
            $message = trans('messages.success.deleted', ['Module' => trans('quote::quotes.name')]);
            $code    = 202;
            $status  = 'success';
            $url     = guard_url('quote/quotes/0');
        } catch (Exception $e) {
            $message = $e->getMessage();
            $code    = 400;
            $status  = 'error';
            $url     = guard_url('quote/quotes/' . $quotes->getRouteKey());
        }
        return compact('message', 'code', 'status', 'url');
    }

    /**
     * Return the form elements as json.
     *
     * @param String   $element
     *
     * @return json
     */
    public function form($element = 'fields')
    {
        $form = new Form();
        return $form->form($element, true);
    }

}
