<?php

namespace Litecms\Quote\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use Litecms\Quote\Http\Requests\QuotesRequest;
use Litecms\Quote\Interfaces\QuotesRepositoryInterface;
use Litecms\Quote\Models\Quotes;

/**
 * Resource controller class for quotes.
 */
class QuotesResourceController extends BaseController
{

    /**
     * Initialize quotes resource controller.
     *
     * @param type QuotesRepositoryInterface $quotes
     *
     * @return null
     */
    public function __construct(QuotesRepositoryInterface $quotes)
    {
        parent::__construct();
        $this->repository = $quotes;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Litecms\Quote\Repositories\Criteria\QuotesResourceCriteria::class);
    }

    /**
     * Display a list of quotes.
     *
     * @return Response
     */
    public function index(QuotesRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\Litecms\Quote\Repositories\Presenter\QuotesPresenter::class)
                ->$function();
        }

        $quotes = $this->repository->paginate();

        return $this->response->setMetaTitle(trans('quote::quotes.names'))
            ->view('quote::quotes.index', true)
            ->data(compact('quotes', 'view'))
            ->output();
    }

    /**
     * Display quotes.
     *
     * @param Request $request
     * @param Model   $quotes
     *
     * @return Response
     */
    public function show(QuotesRequest $request, Quotes $quotes)
    {

        if ($quotes->exists) {
            $view = 'quote::quotes.show';
        } else {
            $view = 'quote::quotes.new';
        }

        return $this->response->setMetaTitle(trans('app.view') . ' ' . trans('quote::quotes.name'))
            ->data(compact('quotes'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new quotes.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(QuotesRequest $request)
    {

        $quotes = $this->repository->newInstance([]);
        return $this->response->setMetaTitle(trans('app.new') . ' ' . trans('quote::quotes.name')) 
            ->view('quote::quotes.create', true) 
            ->data(compact('quotes'))
            ->output();
    }

    /**
     * Create new quotes.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(QuotesRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $quotes                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('quote::quotes.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('quote/quotes/' . $quotes->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/quote/quotes'))
                ->redirect();
        }

    }

    /**
     * Show quotes for editing.
     *
     * @param Request $request
     * @param Model   $quotes
     *
     * @return Response
     */
    public function edit(QuotesRequest $request, Quotes $quotes)
    {
        return $this->response->setMetaTitle(trans('app.edit') . ' ' . trans('quote::quotes.name'))
            ->view('quote::quotes.edit', true)
            ->data(compact('quotes'))
            ->output();
    }

    /**
     * Update the quotes.
     *
     * @param Request $request
     * @param Model   $quotes
     *
     * @return Response
     */
    public function update(QuotesRequest $request, Quotes $quotes)
    {
        try {
            $attributes = $request->all();

            $quotes->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('quote::quotes.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('quote/quotes/' . $quotes->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('quote/quotes/' . $quotes->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the quotes.
     *
     * @param Model   $quotes
     *
     * @return Response
     */
    public function destroy(QuotesRequest $request, Quotes $quotes)
    {
        try {

            $quotes->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('quote::quotes.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('quote/quotes/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('quote/quotes/' . $quotes->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple quotes.
     *
     * @param Model   $quotes
     *
     * @return Response
     */
    public function delete(QuotesRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('quote::quotes.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('quote/quotes'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/quote/quotes'))
                ->redirect();
        }

    }

    /**
     * Restore deleted quotes.
     *
     * @param Model   $quotes
     *
     * @return Response
     */
    public function restore(QuotesRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('quote::quotes.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/quote/quotes'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/quote/quotes/'))
                ->redirect();
        }

    }

}
