<?php

namespace Shopping\Status\Http\Controllers;

use App\Http\Controllers\ResourceController as BaseController;
use Form;
use Shopping\Status\Http\Requests\StatusRequest;
use Shopping\Status\Interfaces\StatusRepositoryInterface;
use Shopping\Status\Models\Status;

/**
 * Resource controller class for status.
 */
class StatusResourceController extends BaseController
{

    /**
     * Initialize status resource controller.
     *
     * @param type StatusRepositoryInterface $status
     *
     * @return null
     */
    public function __construct(StatusRepositoryInterface $status)
    {
        parent::__construct();
        $this->repository = $status;
        $this->repository
            ->pushCriteria(\Litepie\Repository\Criteria\RequestCriteria::class)
            ->pushCriteria(\Shopping\Status\Repositories\Criteria\StatusResourceCriteria::class);
    }

    /**
     * Display a list of status.
     *
     * @return Response
     */
    public function index(StatusRequest $request)
    {
        $view = $this->response->theme->listView();

        if ($this->response->typeIs('json')) {
            $function = camel_case('get-' . $view);
            return $this->repository
                ->setPresenter(\Shopping\Status\Repositories\Presenter\StatusPresenter::class)
                ->$function();
        }

        $statuses = $this->repository->paginate();

        return $this->response->title(trans('status::status.names'))
            ->view('status::status.index', true)
            ->data(compact('statuses'))
            ->output();
    }

    /**
     * Display status.
     *
     * @param Request $request
     * @param Model   $status
     *
     * @return Response
     */
    public function show(StatusRequest $request, Status $status)
    {

        if ($status->exists) {
            $view = 'status::status.show';
        } else {
            $view = 'status::status.new';
        }

        return $this->response->title(trans('app.view') . ' ' . trans('status::status.name'))
            ->data(compact('status'))
            ->view($view, true)
            ->output();
    }

    /**
     * Show the form for creating a new status.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function create(StatusRequest $request)
    {

        $status = $this->repository->newInstance([]);
        return $this->response->title(trans('app.new') . ' ' . trans('status::status.name')) 
            ->view('status::status.create', true) 
            ->data(compact('status'))
            ->output();
    }

    /**
     * Create new status.
     *
     * @param Request $request
     *
     * @return Response
     */
    public function store(StatusRequest $request)
    {
        try {
            $attributes              = $request->all();
            $attributes['user_id']   = user_id();
            $attributes['user_type'] = user_type();
            $status                 = $this->repository->create($attributes);

            return $this->response->message(trans('messages.success.created', ['Module' => trans('status::status.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('status/status/' . $status->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('/status/status'))
                ->redirect();
        }

    }

    /**
     * Show status for editing.
     *
     * @param Request $request
     * @param Model   $status
     *
     * @return Response
     */
    public function edit(StatusRequest $request, Status $status)
    {
        return $this->response->title(trans('app.edit') . ' ' . trans('status::status.name'))
            ->view('status::status.edit', true)
            ->data(compact('status'))
            ->output();
    }

    /**
     * Update the status.
     *
     * @param Request $request
     * @param Model   $status
     *
     * @return Response
     */
    public function update(StatusRequest $request, Status $status)
    {
        try {
            $attributes = $request->all();

            $status->update($attributes);
            return $this->response->message(trans('messages.success.updated', ['Module' => trans('status::status.name')]))
                ->code(204)
                ->status('success')
                ->url(guard_url('status/status/' . $status->getRouteKey()))
                ->redirect();
        } catch (Exception $e) {
            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('status/status/' . $status->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove the status.
     *
     * @param Model   $status
     *
     * @return Response
     */
    public function destroy(StatusRequest $request, Status $status)
    {
        try {

            $status->delete();
            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('status::status.name')]))
                ->code(202)
                ->status('success')
                ->url(guard_url('status/status/0'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->code(400)
                ->status('error')
                ->url(guard_url('status/status/' . $status->getRouteKey()))
                ->redirect();
        }

    }

    /**
     * Remove multiple status.
     *
     * @param Model   $status
     *
     * @return Response
     */
    public function delete(StatusRequest $request, $type)
    {
        try {
            $ids = hashids_decode($request->input('ids'));

            if ($type == 'purge') {
                $this->repository->purge($ids);
            } else {
                $this->repository->delete($ids);
            }

            return $this->response->message(trans('messages.success.deleted', ['Module' => trans('status::status.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('status/status'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/status/status'))
                ->redirect();
        }

    }

    /**
     * Restore deleted statuses.
     *
     * @param Model   $status
     *
     * @return Response
     */
    public function restore(StatusRequest $request)
    {
        try {
            $ids = hashids_decode($request->input('ids'));
            $this->repository->restore($ids);

            return $this->response->message(trans('messages.success.restore', ['Module' => trans('status::status.name')]))
                ->status("success")
                ->code(202)
                ->url(guard_url('/status/status'))
                ->redirect();

        } catch (Exception $e) {

            return $this->response->message($e->getMessage())
                ->status("error")
                ->code(400)
                ->url(guard_url('/status/status/'))
                ->redirect();
        }

    }

}
